/*
 ============================================================================
 Name        : Project.c
 Author      : Dylan Samson
 Version     : 
 Copyright   :
 Description : Real Time Data Collection Project
 ============================================================================
 */

#include <time.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <unistd.h>
#include <termios.h>
#include <pthread.h>
#include <semaphore.h>
#include <rtai_fifos.h>
#include <string.h>

RTIME BaseP;

//structure used to store temperature measurement and time of measurement
typedef struct TempPoint{
	double value_a;
	struct timeval time_a;
}TempPoint;

//structure used to share variables between threads
typedef struct Options{
	int FileChoice;
	int PrintChoice;
	int StopChoice;
	int PeriodicChoice;
	int PauseChoice;
	int Flag;
	int Reset;
	struct timeval StartTime;
	int TimeSet;
	double timeToMeasure;
	double tempToNotify;
	int tempNotifyUnit;
}Options;

sem_t sem1;
sem_t sem2;

void dataCollect(void *ptr);
void getTemp();
void fileWrite(void *ptr);
void userInterface(void *ptr);
void buttonPress(void *ptr);
void monitorTime(void *ptr);
int  kbhit(void);

int main(void) {
	Options Opts;				   //main structure declaration
	pthread_t thread0;			   //thread that will be used for data collection
	volatile unsigned char *option; //bit 0 of this register will be used to check if MAX197 installed
	unsigned char *optionValue;
	int fd = open("/dev/mem", O_RDWR|O_SYNC); //file descriptor for mapping
	double PeriodTemp = 0;					  //used for setting the real time period
	int PeriodCh;							  //stores Y/N for period choice

	system("mkfifo named_pipe >& /dev/null"); //create named pipe
	system("clear");						  
	sem_init(&sem1, 0, 1);					  //initialize semaphores
	sem_init(&sem2, 0, 0);

	//initialize variables of main structure
	Opts.StopChoice = 0;
	Opts.PrintChoice = 0;
	Opts.PeriodicChoice = 0;
	Opts.PauseChoice = 0;
	Opts.TimeSet = 0;
	Opts.tempToNotify = 0;

	//map address of option register
	option = mmap(0, getpagesize(), PROT_READ|PROT_WRITE, MAP_SHARED, fd, 0x22400000);

	//check bit 0 to see if MAX197 ADC is installed
	optionValue = (unsigned char *)option;
	if(*optionValue & 0x01){
		printf("MAX197 Connected\n");
	}
	else{
		printf("MAX197 Not Connected\n");
		pthread_exit(0);
	}

	//check if user would like to start periodic data collection
	printf("\nWould you like to take periodic measurements (Y/N)? ");
	PeriodCh = getchar();

	if(PeriodCh == 'Y'){
		printf("Enter period length in milliseconds? "); //get period value from user
		scanf("%d", &Opts.PeriodicChoice);
		PeriodTemp = Opts.PeriodicChoice;
		PeriodTemp = PeriodTemp * 1000000; //convert to nanoseconds

		BaseP = start_rt_timer(nano2count(PeriodTemp)); //set base period for real time task

		//create thread for data collection
		pthread_create(&thread0, NULL, (void*)&dataCollect, (void *)&Opts);

		pthread_join(thread0, NULL);
	}
	else{
		printf("\n");
	}

	usleep(5000);
	return 0;
}

//finishes setup and manages data collection
void dataCollect(void *ptr){
	Options *Opts = (Options *)ptr;
	int iterations;
	
	//get how long the user would like to run program
	printf("\nHow long in seconds would you like to take measurements? ");
	scanf("%d", &iterations);
	//calculate number of temperature measurements to take
	iterations = ((iterations * 1000) / Opts->PeriodicChoice) + 1;

	//get a time user would like to take a specific measurement
	printf("\nWould you like to take a measurement at a specific time?\n"
			"Specify time in milliseconds (0 = No): ");
	scanf("%lf", &Opts->timeToMeasure);

	//get a temp user would like to be notified of when it is reached
	printf("\nWould you like to be notified at a specific temp?\n"
				"Specify temp in centigrade (0 = No): ");
	scanf("%lf", &Opts->tempToNotify);

	//ask if user would like to save data to files
	Opts->FileChoice = getchar();
	printf("\nWould you like to save these measurements to a file (Y/N)? ");
	Opts->FileChoice = getchar();

	//ask user what kind of files they would like and set choice
	if(Opts->FileChoice == 'Y'){
		printf("What kind of file would you like?\n");
		printf("1 - comma-separated values (.csv)\n");
		printf("2 - general text output (.txt)\n");
		printf("3 - both a .csv and .txt file\n");
		printf("4 - cancel saving of measurments to file\n");
		scanf("%d", &Opts->FileChoice);
	}

	//create thread for writing to output files and printing to terminal
	pthread_t thread1;
	pthread_create(&thread1, NULL, (void *)&fileWrite, (void *)Opts);

	system("clear");

	//start thread for terminal interface and button interface
	pthread_t thread3, thread4;
	pthread_create(&thread3, NULL, (void *)&userInterface, (void *)Opts);
	pthread_create(&thread4, NULL, (void *)&buttonPress, (void*)Opts);

	//initialize real time task and make periodic with specified period
	RT_TASK* rtwrite1 = rt_task_init(nam2num("thrd0"), 0, 512, 256);
	rt_task_make_periodic(rtwrite1, rt_get_time() + nano2count(500000000), BaseP);

	int i;
	gettimeofday(&Opts->StartTime, NULL);
	//data collection loop
	for(i = 0; i < iterations; i++){
		if(Opts->StopChoice == 1){
			pthread_exit(0); //if stop has been selected, abort data collection
		}

		if(Opts->PauseChoice != 1){
			getTemp(); //get temperature measurement
		}
		else{
			if(Opts->PrintChoice == 1){
				printf("\nPaused...\n"); //shows pause if terminal output enabled and paused
			}
		}

		if(Opts->Reset == 1){
			i = 0; //if reset as has be selected, reset data collection
		}
		rt_task_wait_period(); //wait until next period
	}

	//alert user that data collection completed
	printf("\nData Collection Complete!\n");
	usleep(5000);
	pthread_exit(0);
}

//gets a temperature reading from channel 0 of the MAX 197 ADC on the TS-7250
void getTemp(){
	volatile unsigned char *option, *controlValue, *busy; //registers for ADC
	TempPoint *TempMeasure; //structure for temperature measurement
	int fd = open("/dev/mem", O_RDWR|O_SYNC); //file descriptor for mapping
	int named_pipe;
	unsigned short *result;
	struct timeval time;
	unsigned char *busyValue;
	unsigned char *optionValue;

	TempMeasure = (TempPoint*)malloc(sizeof(TempPoint)); //allocate memory

	//map addresses of registers
	option = mmap(0, getpagesize(), PROT_READ|PROT_WRITE, MAP_SHARED, fd, 0x22400000);
	controlValue = mmap(0, getpagesize(), PROT_READ|PROT_WRITE, MAP_SHARED, fd, 0x10F00000);
	busy = mmap(0, getpagesize(), PROT_READ|PROT_WRITE, MAP_SHARED, fd, 0x10800000);

	//open named pipe for writing
	if((named_pipe = open("named_pipe", O_WRONLY)) < 0){
		printf("error while creating pipe\n");
		pthread_exit(0);
	}

	//get time of temperature measurement
	gettimeofday(&TempMeasure->time_a, NULL);

	optionValue = (unsigned char *)option;

	//get sample from ADC Channel 0, Unipolar, 5V Range
	*controlValue = 0x40;

	busyValue = (unsigned char *)busy;

	//wait until conversion complete 
	while(*busyValue & 0x80){ //bit 7 is 1 when complete
		busyValue = (unsigned char *)busy;
		printf("");
	}
	result = (unsigned short *)controlValue;

	//store result in structure as a double
	TempMeasure->value_a = (double)*result;

	//write to pipe to get temperature structure to fileWrite thread
	sem_wait(&sem1);
	if(write(named_pipe, TempMeasure, sizeof(*TempMeasure)) != sizeof(*TempMeasure)){
		printf("error while writing to pipe\n");
		pthread_exit(0);
	}
	sem_post(&sem2);

}

//writes to .csv and .txt (if selected) and displays to terminal (if selected)
void fileWrite(void *ptr){
	Options *Opts = (Options *)ptr;
	TempPoint *TempMeasure;
	TempMeasure = (TempPoint*)malloc(sizeof(TempPoint));
	int named_pipe;
	double tempCel, tempFehr, tempKelv;
	FILE *fp1, *fp2;
	double timeMS;

	//if .csv option selected, open/create .csv file for writing
	if(Opts->FileChoice == 1 || Opts->FileChoice == 3){
		fp1 = fopen("data.csv", "w+");
		fprintf(fp1, "Time, Celsius, Fehrenheit, Kelvin, Flag");
	}

	//if .txt option selected, open/create .txt file for writing
	if(Opts->FileChoice == 2 || Opts->FileChoice == 3){
		fp2 = fopen("data.txt", "w+");
		fprintf(fp2, "Temperature Data\n");
	}

	//open named pipe for reading
	if((named_pipe = open("named_pipe", O_RDONLY)) < 0){
		printf("error while creating pipe\n");
		pthread_exit(0);
	}

	while(1){
		sem_wait(&sem2);
		//if reset selected, then reopen output files erasing pevious contents
		if(Opts->Reset == 1){
			if(Opts->FileChoice == 1 || Opts->FileChoice == 3){
				fclose(fp1);
				fp1 = fopen("data.csv", "w+");
				fprintf(fp1, "Time, Celsius, Fehrenheit, Kelvin, Flag");
			}

			if(Opts->FileChoice == 2 || Opts->FileChoice == 3){
				fclose(fp2);
				fp2 = fopen("data.txt", "w+");
				fprintf(fp2, "Temperature Data\n");
			}
			Opts->TimeSet = 0; //reset start time
			Opts->Reset = 0;   //deselect reset
		}

		//read from named pipe
		if((read(named_pipe, TempMeasure, sizeof(*TempMeasure))) < 0){
			printf("error while reading from pipe\n");
			pthread_exit(0);
		}

		//set start time if timeset is 0 meaning start time has not been set
		if(Opts->TimeSet == 0){
			Opts->StartTime = TempMeasure->time_a;
			if(Opts->timeToMeasure != 0){
				//start monitoring for time specified by user
				pthread_t thread2;
				pthread_create(&thread2, NULL, (void *)&monitorTime, (void *)Opts);
			}
			Opts->TimeSet = 1; //set timeset to show start time has been set
		}

		//convert time to milliseconds
		timeMS = ((TempMeasure->time_a.tv_sec * 1000.0) - (Opts->StartTime.tv_sec * 1000.0));
		timeMS += ((TempMeasure->time_a.tv_usec / 1000.0) - (Opts->StartTime.tv_usec / 1000.0));

		//convert ADC result to Celsius, Fehrenheit, and Kelvin
		tempCel = TempMeasure->value_a;
		tempCel = tempCel * 5000 / 4096;
		tempCel = (tempCel - 500) / 10;
		tempKelv = tempCel + 273.15;
		tempFehr = tempCel * 1.8 + 32;

		//if .csv is selected, perform writing of temperature measurement to data.csv
		if(Opts->FileChoice == 1 || Opts->FileChoice == 3){
			if(Opts->Flag == 1){ //if flag selected, flag measurement
				fprintf(fp1, ", Flag! Comment...");
			}
			fprintf(fp1, "\n%.3f, %.1f, %.1f, %.1f", timeMS, tempCel, tempFehr, tempKelv);
		}

		//if .txt is selected, perform writing of temperature measurement to data.txt
		if(Opts->FileChoice == 2 || Opts->FileChoice == 3){
			if(Opts->Flag == 1){ //if flag selected, flag measurment
				fprintf(fp2, "Flag! Comment...\n");
			}
			fprintf(fp2, "\nTime: %.3f\n", timeMS);
			fprintf(fp2, "Temp (Celsius): %.1f\n", tempCel);
			fprintf(fp2, "Temp (Fehrenheit): %.1f\n", tempFehr);
			fprintf(fp2, "Temp (Kelvin): %.1f\n", tempKelv);
		}

		//if print to terminal enabled, display temperature measurement to terminal
		if(Opts->PrintChoice == 1){
			printf("\nTime: %.3f\n", timeMS);
			printf("Temp (Celsius): %.1f\n", tempCel);
			printf("Temp (Fehrenheit): %.1f\n", tempFehr);
			printf("Temp (Kelvin): %.1f\n", tempKelv);
		}

		//deselect flag
		Opts->Flag = 0;
		sem_post(&sem1);
	}
	usleep(5000);
	free(TempMeasure);
	pthread_exit(0);
}

//displays user interface to terminal and performs actions
void userInterface(void *ptr){
	Options *Opts = (Options *)ptr;
	int menuChoice;
	printf("Collection Started...\n"); //tell user data collection has started
	while(1){ //display options to user during data collection
		printf("\n1 - Pause/Start Collection\n");
		printf("2 - Stop Data Collection\n");
		printf("3 - Flag Previous Measurement\n");
		printf("4 - View Data Collection\n");
		printf("5 - Change Specified Notification Time\n");
		scanf("%d", &menuChoice);

		//if option 1 selected, start/pause data collection
		if(menuChoice == 1){
			Opts->PauseChoice = Opts->PauseChoice ^ (1 << 0);
			if(Opts->PauseChoice == 0){
				printf("\nCollection Started...\n");
			}
			else if(Opts->PauseChoice == 1){
				printf("\nCollection Paused...\n");
			}
		}
		//if option 2 selected, abort data collection
		else if(menuChoice == 2){
			Opts->StopChoice = 1;
			printf("\nData Collection Aborted!\n");
			pthread_exit(0);
		}
		//if option 3 selected, flag previous measurement
		else if(menuChoice == 3){
			Opts->Flag = 1;
			printf("\nPrevious Measurement Flagged!\n");
		}
		//if option 4 selected, print temperature measurements to terminal
		else if(menuChoice == 4){
			Opts->PrintChoice = 1;
			while(!kbhit()){ //wait for keyboard press and then end viewing
			printf("");
			}
			menuChoice = 0;
			Opts->PrintChoice = 0;
		}
		//if option 5 selected, prompt user for new time to be specified
		else if(menuChoice == 5){
			printf("\nWould you like to take a measurement at a specific time?\n"
				   "Specify time in milliseconds (0 = No): ");
			scanf("%lf", &Opts->timeToMeasure);
		}
	}
}

//performs actions of button presses
void buttonPress(void *ptr){
	Options *Opts = (Options *)ptr;
	int fifo, button;

	//open fifo to get button press from kernel module
	fifo = open("/dev/rtf/1", O_RDWR);

	while(1){
		read(fifo, &button, sizeof(int)); //read from fifo
		switch(button){
		case 1: //if B0 pressed, start/pause data collection
			Opts->PauseChoice = Opts->PauseChoice ^ (1 << 0);
			break;
		case 2: //if B1 pressed, abort data collection
			Opts->StopChoice = 1;
			printf("\nData Collection Aborted!\n");
			usleep(100000000);
			break;
		case 3: //if B2 pressed, get temperature measurement
			getTemp();
			break;
		case 4: //if B3 pressed, flag previous measurement
			printf("\nPrevious Measurement Flagged!\n");
			Opts->Flag = 1;
			break;
		case 5: //if B4 pressed, reset data collection
			Opts->Reset = 1;
			break;
		}
	}
}

void monitorTime(void *ptr){
	Options *Opts = (Options *)ptr;
	struct timeval time;
	double timeDiff;

	while(1){ //continually get time of day and compare to user specified time
		gettimeofday(&time, NULL);
		timeDiff = ((time.tv_sec * 1000.0) - (Opts->StartTime.tv_sec * 1000.0));
		timeDiff += ((time.tv_usec / 1000.0) - (Opts->StartTime.tv_usec / 1000.0));
		timeDiff = abs(Opts->timeToMeasure - timeDiff);

		if(timeDiff < 1){ //take temperature measurement if near time specified
			getTemp();
		}
	}
}

//function used to monitor for a keyboard press
//this is a function that is widely available on the internet for this purpose
//I did not create this function myself, I got it from the web address below
//http://cboard.cprogramming.com/c-programming/63166-kbhit-linux.html
int kbhit(void)
{
	struct timeval tv;
	fd_set rdfs;

	tv.tv_sec = 0;
	tv.tv_usec = 0;

	FD_ZERO(&rdfs);
	FD_SET(STDIN_FILENO, &rdfs);

	select(STDIN_FILENO+1, &rdfs, NULL, NULL, &tv);
	return FD_ISSET(STDIN_FILENO, &rdfs);
}
